#include "NetInfo.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	CachedNetInfo gNetInfo;

	DO_EXPORT( bool, PrxNetInfoIsBusy ) ()
	{
		return gNetInfo.IsBusy();
	}

	DO_EXPORT( bool, PrxNetInfoGetLastError ) (ResultCode* result)
	{
		return gNetInfo.GetLastError(result);
	}

	DO_EXPORT( ErrorCode, PrxNetInfoRequestBandwidth ) ()
	{
		return gNetInfo.RequestBandwidth();
	}

	DO_EXPORT( bool, PrxNetInfoGetBandwidth ) (Bandwidth* bandwidth)
	{
		return gNetInfo.GetBandwidth(bandwidth);
	}

	DO_EXPORT( ErrorCode, PrxNetInfoRequestInfo ) ()
	{
		return gNetInfo.RequestNetInfo();
	}

	DO_EXPORT( ErrorCode, PrxNetInfoGetInfo ) (NetInfoBasic* info)
	{
		return gNetInfo.GetNetInfo(info);
	}

	DO_EXPORT( int, PrxNetInfoGetDeviceType ) ()
	{
		return 1;	// Assume a wired connection for PS3 & PS4.
	}

	CachedNetInfo::CachedNetInfo()
		: m_Busy(false)
		, m_LastResult("NetInfo")
	{
	}

	bool CachedNetInfo::ProcessEvent(const Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;
		switch(event.event)
		{
			case Event::netInfoGotBandwidth:
				if(m_FutureBandwidth.hasError())
				{
					m_LastResult.SetResultSCE(m_FutureBandwidth.getError(), true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_NetInfoError);
					handled = true;
					m_Busy = false;
					break;
				}
				else
				{
					m_Bandwidth.downloadBPS = (float)m_FutureBandwidth.get()->download_bps;
					m_Bandwidth.uploadBPS = (float)m_FutureBandwidth.get()->upload_bps;
				}

				Messages::AddMessage(Messages::kNPToolKit_NetInfoGotBandwidth);
				handled = true;
				m_Busy = false;
				break;

			case Event::netInfoGotBasic:					// An event generated when basic network information has been obtained.
				Messages::AddMessage(Messages::kNPToolKit_NetInfoGotBasic);
				handled = true;
				m_Busy = false;
				break;

			case Event::netInfoError:						// An event generated when an error occurs in the net information service.
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_NetInfoError);
				handled = true;
				m_Busy = false;
				break;

			case Event::netInfoGotDetailed:					// An event generated when detailed network information has have been obtained.
				UnityPlugin::Messages::LogWarning("netInfo event not handled: event=%d\n", event.event);
				handled = true;
				m_Busy = false;
				break;

			default:
				break;
		}

		return handled;
	}

	bool CachedNetInfo::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_Busy;
	}

	ErrorCode CachedNetInfo::RequestBandwidth()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		m_FutureBandwidth.reset();

		int ret = NetInfo::Interface::getBandwidthInfo( &m_FutureBandwidth );
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedNetInfo::GetBandwidth(Bandwidth* bandwidth)
	{
		SimpleLock::AutoLock lock(m_Lock);
		*bandwidth = m_Bandwidth;
		return NP_OK;
	}

	ErrorCode CachedNetInfo::RequestNetInfo()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int ret = NetInfo::Interface::getNetInfo(&m_FutureNetStateBasic);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedNetInfo::GetNetInfo(NetInfoBasic* info)
	{
		sce::Toolkit::NP::NetStateBasic* netInfo = m_FutureNetStateBasic.get();
		if(netInfo)
		{
			info->connectionStatus = netInfo->connectionStatus;
			info->ipAddress = netInfo->ipAddress;
			info->natType = netInfo->natInfo.nat_type;
			info->natStunStatus = netInfo->natInfo.stun_status;
			info->natMappedAddr = netInfo->natInfo.mapped_addr.s_addr;
		}
		else
		{
			memset(info, 0, sizeof(NetInfoBasic));
			info->ipAddress = "0.0.0.0";
		}
		return NP_OK;
	}
}
